package nl.moj.test.junit;

import java.util.logging.Level;
import java.util.logging.Logger;

import nl.moj.gamerules.MoJ2006CompetitionGameRules;
import nl.moj.model.Assignment;
import nl.moj.model.GameRules;
import nl.moj.model.Team;
import nl.moj.round.TeamImpl;
import nl.moj.test.junit.dummy.DummyAssignment;
import nl.moj.test.junit.dummy.DummyClock;
import nl.moj.test.junit.dummy.DummyOperation;
import nl.moj.test.junit.dummy.DummyState;
import nl.moj.test.junit.dummy.DummyWorkspace;
import junit.framework.Assert;
import junit.framework.TestCase;

/**
 * Tests the basic properties of TeamImpl and the login/logout behavior.
 * A test over a complete round is done in the RoundTest.
 */
public class TeamTest extends TestCase {

	private static final String TEAMNAME="TestTeam";
	private static final String TEAMDISPLAYNAME="Test Team";
	private static final String PASSWORD="test";

	static {
		// Disable Logging.
		Logger.getLogger("").setLevel(Level.OFF);
	}

	private Team testTeam;
	private DummyState myState;
	private DummyWorkspace myWorkspace;
	private DummyClock myClock;
	private GameRules myGameRules;
	private Assignment myAssignment;
	
    public TeamTest(String arg0) {
        super(arg0);
    }
    protected void setUp() throws Exception {
    	myState=new DummyState();
    	myClock=new DummyClock();
    	myGameRules=new MoJ2006CompetitionGameRules(myClock);
    	myAssignment=new DummyAssignment();
		TeamImpl testTeamImpl=new TeamImpl(TEAMNAME,TEAMDISPLAYNAME,PASSWORD,myAssignment,myGameRules,myState,null);
    	myWorkspace=new DummyWorkspace(testTeamImpl);
		testTeamImpl.setWorkspace(myWorkspace);
        testTeam=testTeamImpl; 
    }
    
    public void testProperties() {
    	Assert.assertEquals("Name",TEAMNAME,testTeam.getName());
		Assert.assertEquals("DisplayName",TEAMDISPLAYNAME,testTeam.getDisplayName());
		Assert.assertFalse("Online(1)",testTeam.isOnline());
		Assert.assertTrue("Valid Password",testTeam.isValidPassword(PASSWORD));
		Assert.assertTrue("Online(2)",testTeam.isOnline());
		Assert.assertFalse("Valid Password",testTeam.isValidPassword(null));
		Assert.assertFalse("Online(3)",testTeam.isOnline());
		Assert.assertEquals(testTeam.getClock().getDuration(),myClock.getDuration());
		//
		for (int m=0;m<30;m++) {
			for (int s=0;s<60;s++) {
				myClock.setTime(m,s);
				Assert.assertEquals(testTeam.getClock().getSecondsRemaining(),myClock.getDuration()-myClock.getSecondsPassed());
				Assert.assertEquals(testTeam.getTheoreticalScore(),400+myClock.getDuration()-myClock.getSecondsPassed());
			}
		}
		//
    }
    
    /**
     * This tests for a rather sneaky bug that appears/ed when running with a workspace server.
     * When a submit operation is executed there are 2 moments that check if the game is still running.
     * First when doing doOperation and secondly when the operation is queued. If the round ends within
     * that interval the following happens:
     * - SUBMITTED does not get written into the state file.
     * - SCORE is written into the state file with MAX points (!).
     * The score board notes this inconsistency and fails.
     * The solution is only to check once if the game is still running, on the doOperation call.
     */
    
    public void testRatherSubtileSubmitTimingBug() throws Exception {
        myClock.start();
        myClock.setTime(29,59); // One second remaining.
        testTeam.doOperation(new DummyOperation(true,true,1000),null);
        myClock.setTime(30,0); // Time up.
        myClock.stop();
        Thread.sleep(1500);
        assertEquals(401,testTeam.getGameRules().getFinalScore(testTeam));
    }
    
    public void testSubmitAfterEnd() throws Exception {
        myClock.start();
        myClock.setTime(30,0); // Time up.
        myClock.stop();
        testTeam.doOperation(new DummyOperation(true,true,1000),null);
        Thread.sleep(1500);
        assertEquals(0,testTeam.getGameRules().getFinalScore(testTeam));    
    }

    public void testSubmitBeforeEnd() throws Exception {
        myClock.start();
        myClock.setTime(15,0); 
        testTeam.doOperation(new DummyOperation(true,true,1000),null);
        myClock.setTime(14,0); 
        Thread.sleep(1500);
        assertEquals(400+15*60,testTeam.getGameRules().getFinalScore(testTeam));    
    }
    
    public void testDoubleSubmit() throws Exception {
        myClock.start();
        myClock.setTime(15,0); 
        testTeam.doOperation(new DummyOperation(true,true,1000),null);
        Thread.sleep(1500);
        myClock.setTime(14,0); 
        testTeam.doOperation(new DummyOperation(true,true,1000),null);
        Thread.sleep(1500);
        assertEquals(400+15*60,testTeam.getGameRules().getFinalScore(testTeam));    
    }    
    
    
	public static void main(String[] args) {
		junit.textui.TestRunner.run(TeamTest.class);
	}

}
