package nl.moj.test.junit;

import java.util.logging.Level;
import java.util.logging.Logger;

import nl.moj.gamerules.CompetitionGameRules;
import nl.moj.model.Assignment;
import nl.moj.model.GameRules;
import nl.moj.model.Operation;
import nl.moj.model.Round;
import nl.moj.model.State;
import nl.moj.model.Team;
import nl.moj.operation.ContextImpl;
import nl.moj.round.RoundImpl;
import nl.moj.round.TeamImpl;
import nl.moj.test.junit.dummy.DummyAssignment;
import nl.moj.test.junit.dummy.DummyClock;
import nl.moj.test.junit.dummy.DummyState;
import nl.moj.test.junit.dummy.DummyWorkspace;
import junit.framework.Assert;
import junit.framework.TestCase;

/**
 * Tests RoundImpl and a part of TeamImpl.  
 */

public class RoundImplTest extends TestCase {

	private static final String TESTROUND="TestRound";
	private static final String TESTTEAM="TestTeam";
	private static final String PASSWORD="welkom";
	private static final int DURATION=30;
	private static final int MINUTETOSUBMIT=10;
	private static final int EXPECTEDSCORE=(DURATION-MINUTETOSUBMIT)*60;

	private State.Writer myState;
	private DummyClock myClock;
	private GameRules myGameRules;
	private Operation[] myOps;
	private Round myRound;
	private Assignment myAssignment;
	
	static {
		// Disable Logging.
		Logger.getLogger("").setLevel(Level.OFF);
	}

    public RoundImplTest(String arg0) {
        super(arg0);
    }

    protected void setUp() throws Exception {
    	myState=new DummyState();
    	myClock=new DummyClock();
    	myGameRules=new CompetitionGameRules(myClock);
    	myAssignment=new DummyAssignment();
    	myOps=myAssignment.getOperations();
        myRound=new RoundImpl(myAssignment,myGameRules);
        TeamImpl tm=new TeamImpl(TESTTEAM,TESTTEAM,PASSWORD,myAssignment,myGameRules,myState,null);
        tm.setWorkspace(new DummyWorkspace(tm));
        myRound.addTeam(tm);        
    }

    public void testGetName() {
    	Assert.assertEquals("getName",TESTROUND,myRound.getAssignment().getName());
    }

    public void testAddTeam() {
    	TeamImpl tm=new TeamImpl("A","A","A",myAssignment,myGameRules,myState,null);
        tm.setWorkspace(new DummyWorkspace(tm));
    	int cnt=myRound.getTeamCount();
    	myRound.addTeam(tm);
    	Assert.assertEquals("SizeInc",myRound.getTeamCount(),cnt+1);
    	Assert.assertEquals("SameTeam",myRound.getTeamByName("A"),tm);
    }

/*    public void testDisqualifyTeam() {
		Team tm=myRound.getTeamByName(TESTTEAM);
		myRound.disqualifyTeam(tm);
		Assert.assertTrue("Disqualified",tm.isWaiting());
		testStart();
		myRound.disqualifyTeam(tm);
    	Assert.assertTrue("Disqualified",tm.isDisqualified());
    }
*/    

    public void testRemoveTeam() {
		int cnt=myRound.getTeamCount();
		myRound.removeTeam(myRound.getTeamByName(TESTTEAM));
		Assert.assertEquals("SizeInc",cnt-1,myRound.getTeamCount());
    }


    public void testGetAllTeams() {
    	Team[] tm=myRound.getAllTeams();
    	Assert.assertEquals(tm.length,1);
		Assert.assertEquals(tm[0],myRound.getTeamByName(TESTTEAM));
    }

    public void testGetTeamByName() {
    	TeamImpl tm=new TeamImpl("A","A","A",myAssignment,myGameRules,myState,null);
        tm.setWorkspace(new DummyWorkspace(tm));
		myRound.addTeam(tm);
		Assert.assertEquals("SameTeam",myRound.getTeamByName("A"),tm);    	
    }

    public void testGetTeamsOnline() {
    	TeamImpl tm=new TeamImpl("A","A","A",myAssignment,myGameRules,myState,null);
        tm.setWorkspace(new DummyWorkspace(tm));
		myRound.addTeam(tm);
		Assert.assertEquals(0,myRound.getTeamsOnline());
		Assert.assertTrue(tm.isValidPassword("A"));
		Assert.assertEquals(1,myRound.getTeamsOnline());
		Assert.assertTrue(myRound.getTeamByName(TESTTEAM).isValidPassword(PASSWORD));
		Assert.assertEquals(2,myRound.getTeamsOnline());
    }

    protected Team createTeam(String s1,String s2,String s3) {
    	TeamImpl tm=new TeamImpl(s1,s2,s3,myAssignment,myGameRules,myState,null);
        tm.setWorkspace(new DummyWorkspace(tm));
        return tm;
    }
    
    public void testGetTeamCount() {
    	Team[] tm=new Team[] {
    		createTeam("A","A","A"),
    		createTeam("B","B","B"),
    		createTeam("C","C","C"),
    		createTeam("D","D","D"),
    	};
    	Assert.assertEquals(1,myRound.getTeamCount());
    	for (int t=0;t<tm.length;t++) {
    		myRound.addTeam(tm[t]);
			Assert.assertEquals(2+t,myRound.getTeamCount());
    	}
		for (int t=0;t<tm.length;t++) {
			myRound.removeTeam(tm[t]);
			Assert.assertEquals(tm.length-t,myRound.getTeamCount());
		}
    }
    
	/** Tests a time up */
    public void testClockTimeUp() {
    	myClock.start();
    	for (int t=0;t<=30;t++) {
    		myClock.setTime(t,0);
    		myClock.fireMinutePassed(t);  
    	}
    	Assert.assertTrue("Finished",myRound.isFinished());
    	Assert.assertTrue(myRound.getTeamByName(TESTTEAM).isFinished());
    	//Assert.assertTrue(myRound.getTeamByName(TESTTEAM).isScoreAvailable());
    	Assert.assertEquals("Score",0,myRound.getTeamByName(TESTTEAM).getFinalScore());
    }
    
	/** Tests a submission that succeeds */
    public void testClockSubmit() {
    	myClock.start();
		for (int t=0;t<=DURATION;t++) {
			myClock.setTime(t,0);			
			myClock.fireMinutePassed(DURATION-t);
			if (t==MINUTETOSUBMIT) {
				Assert.assertTrue(myRound.getTeamByName(TESTTEAM).isPlaying());
				myRound.getTeamByName(TESTTEAM).doOperation(myOps[1],new ContextImpl("a","a",-1));
				while (myRound.getTeamByName(TESTTEAM).isPerformingOperation()) {
					try { Thread.sleep(100); } catch (InterruptedException ex) { }
				}
			}
		}
		Assert.assertTrue("Finished",myRound.isFinished());
		Assert.assertEquals("Score",EXPECTEDSCORE,myRound.getTeamByName(TESTTEAM).getFinalScore());
    }

	/** Tests a submission that fails */
	public void testClockSubmitFail() {
		myClock.start();
		for (int t=0;t<=DURATION;t++) {
			myClock.setTime(t,0);			
			myClock.fireMinutePassed(DURATION-t);
			if (t==MINUTETOSUBMIT) {
				Assert.assertTrue(myRound.getTeamByName(TESTTEAM).isPlaying());
				myRound.getTeamByName(TESTTEAM).doOperation(myOps[2],new ContextImpl("a","a",-1));
				while (myRound.getTeamByName(TESTTEAM).isPerformingOperation()) {
					try { Thread.sleep(100); } catch (InterruptedException ex) { }
				}
			}
		}
		Assert.assertTrue("Finished",myRound.isFinished());
		Assert.assertEquals("Score",0,myRound.getTeamByName(TESTTEAM).getFinalScore());
	}    
	
	public void testGetOperations() {
		Operation[] op=myRound.getAssignment().getOperations();
		Assert.assertEquals(myOps.length,op.length);
	}

	public static void main(String[] args) {
		junit.textui.TestRunner.run(RoundImplTest.class);
	}


}
