package nl.moj.assignment;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.MissingResourceException;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;

import nl.ctrlaltdev.ioc.ApplicationBuilder;
import nl.ctrlaltdev.util.Tool;
import nl.moj.model.Assignment;
import nl.moj.model.Operation;
import nl.moj.model.Tester;
import nl.moj.model.Tester.SecurityDelegate;
import nl.moj.security.DefaultSecurityDelegate;
import nl.moj.security.SandboxSecurityManager;
import nl.moj.test.JarClassLoader;

/**
 * LegacyPropertyFileAssignment : reads the assignment configuration from the old property file format.
 * Valid name-value pairs are : 
 * NAME=name of the assignment 
 * ASSIGNMENT=name of jar file containing the java, txt and mf files.
 * OPERATIONS=comma separated list of names of operations
 * OPERATIONS.name=class of the operation to load and configure.
 * OPERATIONS.name.xxxx=custom parameters as defined by operation class.
 * SECURITYDELEGATE=class name of security delegate (optional)
 * SECURITYDELEGATE.JAR=jar file containing the security delegate (optional)
 * MONOSPACED=if TRUE the .txt files are rendered in a monospaced font. (overridden by manifest-file)
 * EDITABLE=comma separated list of user editable files. (overridden by manifest-file)
 * @deprecated Use JarFileAssignment
 */
public class LegacyPropertyFileAssignment implements Assignment {

	private static final String EXT_JAVA=".java";
	private static final String EXT_TXT=".txt";
	private static final String EXT_MF=".mf";

	private static class ResourceBundleConfiguration implements Operation.Configuration {
		private ResourceBundle res;
		private String base; 
		private File parent;
		public ResourceBundleConfiguration(File parent,ResourceBundle res,String base) {
			this.res=res;
			this.base=base;
			this.parent=parent;
		}
        public String getParameter(Operation which, String name) {
        	try {
        		return res.getString(base+"."+name);
        	} catch (MissingResourceException ex) {
            	return null;
        	}
        }
        /**
         * @see nl.moj.model.Operation.Configuration#getParentDir(nl.moj.model.Operation)
         */
        public File getParentDir(Operation which) {
            return parent;
        }


	}

	private ResourceBundle myAssignment;
	//private ProcessPool myPool;
	private JarFile mySourceJar;
	private Operation[] myOps;
	private String[] editableFiles;
	private boolean textFileIsMonospaced;
	private List textFiles=new ArrayList();
	private List javaFiles=new ArrayList();
	private ApplicationBuilder parent;
	protected Tester.SecurityDelegate delegate;
	private File base;

	public LegacyPropertyFileAssignment(File src,ApplicationBuilder bld) throws IOException {
		this(src.getParentFile(),new PropertyResourceBundle(new FileInputStream(src)),bld);		
	} 
	
	protected String getString(String name,String defaultValue) {
		try {
			return myAssignment.getString(name);
		} catch (MissingResourceException ex) {
			return defaultValue;
		}
	}

	public LegacyPropertyFileAssignment(File base,ResourceBundle assignment,ApplicationBuilder bld) throws IOException {
		this.base=base;
		myAssignment=assignment;
		//
		File assignmentFile=new File(base,myAssignment.getString("ASSIGNMENT"));
		if (!assignmentFile.exists()) {
			//
			// Legacy Legacy 
			//
			assignmentFile=new File(myAssignment.getString("ASSIGNMENT"));
			if (!assignmentFile.exists()) {
				//
				throw new IOException("Assignment file '"+assignmentFile+"' does not exist.");
				//
			}
		}
		mySourceJar=new JarFile(assignmentFile);
		parent=bld;
		//
		Manifest mf=mySourceJar.getManifest();
		if (mf==null) throw new IOException("Missing Manifest file.");
		Attributes a=mf.getMainAttributes();
		if (a==null) throw new IOException("Missing MainAttributes in Manifest File.");
		String val=a.getValue("Editable");
		if (val==null) {
			val=getString("EDITABLE",null);
			if (val==null) {
				throw new IOException("'Editable' entry in MainAttributes is NULL - missing CRLF at the end ?");
			}
		}
		editableFiles=Tool.cut(val," ");
		//
		String securityDelegate=getString("SECURITYDELEGATE",null);
		if (securityDelegate==null) {
			//
			Logger.getLogger("Assignment").warning("'SECURITYDELEGATE' not set. Using default.");
			delegate=new DefaultSecurityDelegate();
			//
		} else try {
			//
			ClassLoader cl=LegacyPropertyFileAssignment.class.getClassLoader();
			String securityDelegateJar=getString("SECURITYDELEGATE.JAR",null);
			if (securityDelegateJar!=null) {
				JarFile secJar=new JarFile(new File(base,securityDelegateJar)); 
				cl=new JarClassLoader(secJar,cl);
			}
			//
			ApplicationBuilder ab=new ApplicationBuilder(parent);
			ab.register(Manifest.class,mf);
			ab.build(new Class[] {cl.loadClass(securityDelegate)});
			delegate=(Tester.SecurityDelegate)ab.get(Tester.SecurityDelegate.class);
			if (delegate==null) throw new NullPointerException("Error getting '"+securityDelegate+"' as Tester.SecurityDelegate");
			//
		} catch (Exception ex) {
			throw new IOException("Failed instantiating '"+securityDelegate+"' : "+ex);
		} 		
		//
		registerAssignmentWithSecurityManager();		
		//
		String monospaced=a.getValue("Monospaced");
		if (monospaced==null) monospaced=getString("MONOSPACED","FALSE");
		textFileIsMonospaced="TRUE".equalsIgnoreCase(monospaced);
		//
		Enumeration entries=mySourceJar.entries();
		while (entries.hasMoreElements()) {
			JarEntry e=(JarEntry)entries.nextElement();
			String name=e.getName();
			if (name.endsWith(EXT_JAVA)) {
				javaFiles.add(name);
			} else if (name.endsWith(EXT_TXT)) {
				textFiles.add(name);
			} else if (name.endsWith(EXT_MF)) {
				// metainf.mf
			} else if (name.endsWith("/")) {
				// directory.
			} else throw new IOException("Unexpected file : "+name);
		}
		//
	}
	
	protected void registerAssignmentWithSecurityManager() {
		SandboxSecurityManager ssm=(SandboxSecurityManager)System.getSecurityManager();
		ssm.registerAssignment(this);		
	}	

	public String getDisplayName() { return getName(); }
	
	public String getName() {
		return myAssignment.getString("NAME");
	}
	public String[] getDescriptionFileNames() {
		return (String[])textFiles.toArray(new String[textFiles.size()]);
	}
	public String[] getEditableFileNames() {
		return (String[])Tool.copy(editableFiles);
	}
	public String[] getSourceCodeFileNames() {
		return (String[])javaFiles.toArray(new String[javaFiles.size()]);
	}
	public boolean isDescriptionRenderedInMonospaceFont() {
		return textFileIsMonospaced;
	}
	public InputStream getAssignmentFileData(String name) throws IOException {
		ZipEntry ze=mySourceJar.getEntry(name);
		if (ze==null) throw new NullPointerException("ZipEntry not found for "+name); 
		return mySourceJar.getInputStream(ze);
	}

	public Operation[] getOperations() {
		if (myOps==null) {
			String[] opNames=Tool.cut(myAssignment.getString("OPERATIONS"),",");
			myOps=new Operation[opNames.length];		
			for (int t=0;t<myOps.length;t++) {
				myOps[t]=buildOperation(myAssignment,opNames[t]);
			}			
		}
		return (Operation[])Tool.copy(myOps);
	}
	
	/** creates the operations */
	private Operation buildOperation(ResourceBundle res,String name) {
		String type=res.getString("OPERATIONS."+name);
		Operation result=null;
		try {
			Operation.Configuration cfg=new ResourceBundleConfiguration(base,res,"OPERATIONS."+name);
			//
			ApplicationBuilder ab=new ApplicationBuilder(parent);
			ab.register(Operation.Configuration.class,cfg);
			//
			Class c=Class.forName(type);
			ab.build(new Class[] { c });
			result=(Operation)ab.get(c);
			//
		} catch (Exception ex) {
			throw new RuntimeException("Unable to create Operation "+name+".",ex);
		}
		return result;
	}	

	public SecurityDelegate getSecurityDelegate() {
        return delegate;
    }

	public String getAuthor() { return null; }
	public byte[] getIcon() { return null; }
	public byte[] getSponsorImage() { return null; }	
	
	public String getSubmitClass() {
		throw new RuntimeException("Not implemented/not used.");
	}
	public int getSubmitClassTimeout() { 
		throw new RuntimeException("Not implemented/not used.");
	}
	public String getTestClass() {
		throw new RuntimeException("Not implemented/not used.");
	}
	public int getTestClassTimeout() {
		throw new RuntimeException("Not implemented/not used.");		
	}
	public int getDuration() {
		return 30;
	}
	
}
