package nl.moj.test.junit;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutput;
import java.io.DataOutputStream;

import nl.moj.client.anim.Anim;
import nl.moj.client.io.AbstractMessage;
import nl.moj.client.io.ActionMessageImpl;
import nl.moj.client.io.AddActionMessageImpl;
import nl.moj.client.io.AnimationMessageImpl;
import nl.moj.client.io.AssignmentMessageImpl;
import nl.moj.client.io.ConsoleMessageImpl;
import nl.moj.client.io.EditorMessageImpl;
import nl.moj.client.io.GoodbyeMessageImpl;
import nl.moj.client.io.HelloMessageImpl;
import nl.moj.client.io.LogonFailureMessageImpl;
import nl.moj.client.io.Message;
import nl.moj.client.io.MessageFactory;
import nl.moj.client.io.ProtocolVersionMismatchMessageImpl;
import nl.moj.client.io.TestSetMessageImpl;
import nl.moj.client.io.UpdateClientStatisticsMessageImpl;
import nl.moj.model.Assignment;
import nl.moj.test.client.LayeredAnimTestIU;
import nl.moj.test.junit.dummy.DummyAssignment;
import junit.framework.Assert;
import junit.framework.TestCase;

/**
 * Checks the correct serialisation/deserialisation of the messages
 * and the factory.
 */

public class ProtocolTest extends TestCase {

	protected DataInput writeAndReadBack(AbstractMessage msg) throws Exception {
		// 
		ByteArrayOutputStream baos=new ByteArrayOutputStream();
		DataOutput out=new DataOutputStream(baos);
		msg.write(out);
		return new DataInputStream(new ByteArrayInputStream(baos.toByteArray()));
		//
	}

	public void testActionMessage() throws Exception {
		//
		String file="file.txt";
		String contents="blah blah";
		String action="doIt";
		int idx=1;
		int count=99;
		//
		DataInput in=writeAndReadBack(new ActionMessageImpl(file,contents,action,idx,count));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Action);
		Assert.assertEquals(file,((Message.Action)o).getFileName());
		Assert.assertEquals(contents,((Message.Action)o).getContents());
		Assert.assertEquals(action,((Message.Action)o).getAction());
		Assert.assertEquals(idx,((Message.Action)o).getIndex());
		Assert.assertEquals(count,((Message.Action)o).getKeyStrokes());
		//		
	}
	
	public void testAddActionMessage() throws Exception {
		String action="doIt";
		String tooltip="blah etc.";
		boolean confirm=true;
		//
		DataInput in=writeAndReadBack(new AddActionMessageImpl(action,confirm,tooltip));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.AddAction);
		Assert.assertEquals(action,((Message.AddAction)o).getAction());
		Assert.assertEquals(confirm,((Message.AddAction)o).mustConfirm());
		Assert.assertEquals(tooltip,((Message.AddAction)o).getToolTip());
		//		
	}
	
	public void testConsoleMessage() throws Exception {
		String console="doIt";
		String content="blah etc.";
		//
		DataInput in=writeAndReadBack(new ConsoleMessageImpl(console,content));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Console);
		Assert.assertEquals(console,((Message.Console)o).getConsole());
		Assert.assertEquals(content,((Message.Console)o).getContent());
		//		
	}	

	public void testEditorMessage() throws Exception {
		String file="doIt";
		String content="blah etc.";
		boolean java=true;
		boolean readonly=false;
		boolean mono=true;
		//
		DataInput in=writeAndReadBack(new EditorMessageImpl(file,content,java,readonly,mono));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Editor);
		Assert.assertEquals(file,((Message.Editor)o).getFileName());
		Assert.assertEquals(content,((Message.Editor)o).getContents());
		Assert.assertEquals(java,((Message.Editor)o).isJava());
		Assert.assertEquals(readonly,((Message.Editor)o).isReadOnly());
		Assert.assertEquals(mono,((Message.Editor)o).isMonospaced());
		//		
	}	
	
	public void testHelloMessage() throws Exception {
		String user="user";
		String pass="pass";
		//
		DataInput in=writeAndReadBack(new HelloMessageImpl(user,pass));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Hello);
		Assert.assertEquals(user,((Message.Hello)o).getTeamName());
		Assert.assertEquals(pass,((Message.Hello)o).getPassword());
		Assert.assertEquals(Message.PROTOCOLVERSION,((Message.Hello)o).getProtocolVersion());
		//		
	}

	public void testLogonFailureMessage() throws Exception {
		//
		DataInput in=writeAndReadBack(new LogonFailureMessageImpl());
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message);
		Assert.assertEquals(((Message)o).getType(),Message.MSG_UNKNOWNUSERPASSWORD);
		//		
	}
	
	public void testProtocolMismatch() throws Exception {
		//
		DataInput in=writeAndReadBack(new ProtocolVersionMismatchMessageImpl());
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message);
		Assert.assertEquals(((Message)o).getType(),Message.MSG_PROTOCOLVERSIONMISMATCH);
		//		
	}	
	
	public void testTestSetMessage() throws Exception {
		String[] names=new String[] { "a","b","c" };
		String[] descriptions=new String[] { "e","f","g" };
		//
		DataInput in=writeAndReadBack(new TestSetMessageImpl(names,descriptions));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.TestSet);
		Assert.assertEquals(names.length,((Message.TestSet)o).getCount());
		Assert.assertEquals(descriptions.length,((Message.TestSet)o).getCount());
		for (int t=0;t<names.length;t++) {
			Assert.assertEquals(names[t],((Message.TestSet)o).getName(t));
			Assert.assertEquals(descriptions[t],((Message.TestSet)o).getDescription(t));
		}
		//
	}
	
	public void testUpdateClientStatisticsMessage() throws Exception {
		//
		int finalScore=100;
		int max=3600;
		int remain=3500;
		int state=0;
		int teamsOnline=5;
		int teamCount=8;
		int[] results=new int[] { 1,-1,0 };
		//
		DataInput in=writeAndReadBack(new UpdateClientStatisticsMessageImpl(finalScore,max,remain,state,teamsOnline,teamCount,results));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.UpdateClientStatistics);
		Assert.assertEquals(finalScore,((Message.UpdateClientStatistics)o).getFinalScore());
		Assert.assertEquals(max,((Message.UpdateClientStatistics)o).getMaxSeconds());
		Assert.assertEquals(remain,((Message.UpdateClientStatistics)o).getSecondsRemaining());
		Assert.assertEquals(state,((Message.UpdateClientStatistics)o).getState());
		Assert.assertEquals(teamsOnline,((Message.UpdateClientStatistics)o).getTeamsOnline());
		Assert.assertEquals(teamCount,((Message.UpdateClientStatistics)o).getTeamCount());
		int[] tmp=((Message.UpdateClientStatistics)o).getTestResults();
		Assert.assertEquals(results.length,tmp.length);
		for (int t=0;t<tmp.length;t++) {
			Assert.assertEquals(results[t],tmp[t]);
		}
		//		
	}	
	
	public void testMessageSequence() throws Exception {
		int last=-1;
		for (int t=0;t<100;t++) {
			DataInput in=writeAndReadBack(new LogonFailureMessageImpl());
			Message m=new MessageFactory().createMessage(in);
			if (last>=0) Assert.assertEquals(last+1,m.getId());
			last=m.getId();
		}
	}
	
	public void testGoodbye() throws Exception {
		DataInput in=writeAndReadBack(new GoodbyeMessageImpl());
		Object o=new MessageFactory().createMessage(in);
		Assert.assertTrue(o instanceof Message.GoodBye);
	}
	
	public void testAnim() throws Exception {
		int nr=1;
		Anim a=LayeredAnimTestIU.createAnim();
		//
		DataInput in=writeAndReadBack(new AnimationMessageImpl(nr,a));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Animation);
		Assert.assertEquals(nr,((Message.Animation)o).getTest());
		Assert.assertNotNull(((Message.Animation)o).getAnimation());
		Assert.assertEquals(a.getFrameCount(),((Message.Animation)o).getAnimation().getFrameCount());
		//
	}
	
	public void testAssignment() throws Exception {
		//
		Assignment a=new DummyAssignment() {
			public String getDisplayName() { return "TestName"; }
			public String getAuthor() { return "TestAuthor"; }
			public byte[] getIcon() { return new byte[] { 0,1,2,3,4 }; }
			public byte[] getSponsorImage() { return new byte[] { 0,1,2,3,4,5 }; }
		};
		//
		DataInput in=writeAndReadBack(new AssignmentMessageImpl(a));
		Object o=new MessageFactory().createMessage(in);
		//
		Assert.assertTrue(o instanceof Message.Assignment);
		Assert.assertEquals(a.getAuthor(),((Message.Assignment)o).getAuthor());
		Assert.assertEquals(a.getDisplayName(),((Message.Assignment)o).getName());
		Assert.assertEquals(a.getIcon().length,((Message.Assignment)o).getIcon().length);
		Assert.assertEquals(a.getSponsorImage().length,((Message.Assignment)o).getSponsorImage().length);
	}


	public static void main(String[] args) {
		junit.textui.TestRunner.run(ProtocolTest.class);
	}
	
}
