package nl.moj.test.junit;

import java.util.logging.Level;
import java.util.logging.Logger;

import nl.moj.process.ProcessPool;
import junit.framework.Assert;
import junit.framework.TestCase;

/**
 * Tests the ProcessPool by starting a number of processes and doing the
 * book keeping on how long and how many processes should run.
 * @author E.Hooijmeijer 
 */

public class ProcessPoolTest extends TestCase {

	/** simple process that waits for a specified time and optionally throws an exception */
	private static class DummyProcess implements Runnable {
		private int myLenght;
		private boolean myFail;
		public DummyProcess(int length,boolean fail) {
			myLenght=length;
			myFail=fail;
		}
        public void run() {
			try {
				Thread.sleep(myLenght);
			} catch (InterruptedException ex) {
				// Meuh ?				
			} finally {
				if (myFail) throw new RuntimeException(">> TEST EXCEPTION << Please Ignore.");
			}
        }
	}
	
	/** ProcessListener that does the actual book keeping for the test. */
	private static class ProcessListener implements ProcessPool.ProcessListener {
		private int running;
		private int maxRunning;
		private int finished;
		private int queued;
		public synchronized void queued(Runnable r) {
			queued++;
		}
		public synchronized void executing(Runnable r) {
			queued--;
			running++;
			if (maxRunning<running) maxRunning++;
		}
		public synchronized void complete(Runnable r) {
			running--;
			finished++;
		}
		public synchronized int getMaxRunning() { return maxRunning; }
		public synchronized int getQueued() { return queued; }
		public synchronized int getRunning() { return running; }
		public synchronized int getFinished() { return finished; }

	}
	
	static {
		// Disable Logging.
		Logger.getLogger("").setLevel(Level.OFF);
	}
	

	//
	// Actual Tests
	//

    public ProcessPoolTest(String arg0) {
        super(arg0);
    }

	/** tests a ProcessPool with just one thread. */
    public void testExecuteOneThread() {
		ProcessPool current=new ProcessPool(1);
		ProcessListener pl=new ProcessListener();
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		// Wait til everything should be finished.
		try { Thread.sleep(5000); } catch (InterruptedException ex) { }
		Assert.assertEquals("No tasks queued",0,pl.getQueued());
		Assert.assertEquals("No tasks running",0,pl.getRunning());
		Assert.assertEquals("Max One Thread Running",1,pl.getMaxRunning());
		Assert.assertEquals("All Tasks Finished",4,pl.getFinished());
    }

	/** tests a ProcessPool with two threads. */
	public void testExecuteTwoThread() {
		ProcessPool current=new ProcessPool(2);
		ProcessListener pl=new ProcessListener();
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		// Wait til everything should be finished.
		try { Thread.sleep(3000); } catch (InterruptedException ex) { }
		Assert.assertEquals("No tasks queued",0,pl.getQueued());
		Assert.assertEquals("No tasks running",0,pl.getRunning());
		Assert.assertEquals("Max One Thread Running",2,pl.getMaxRunning());
		Assert.assertEquals("All Tasks Finished",4,pl.getFinished());
	}    


	/** tests a ProcessPool with four threads. */
	public void testExecuteFourThread() {
		ProcessPool current=new ProcessPool(4);
		ProcessListener pl=new ProcessListener();
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		current.execute(new DummyProcess(1000,false),pl);
		// Wait til everything should be finished.
		try { Thread.sleep(2000); } catch (InterruptedException ex) { }
		Assert.assertEquals("No tasks queued",0,pl.getQueued());
		Assert.assertEquals("No tasks running",0,pl.getRunning());
		Assert.assertEquals("Max One Thread Running",4,pl.getMaxRunning());
		Assert.assertEquals("All Tasks Finished",4,pl.getFinished());
	} 
	
	/** tests a ProcessPool with two threads and processes that fail. */
	public void testExecuteFailThread() {
		ProcessPool current=new ProcessPool(2);
		ProcessListener pl=new ProcessListener();
		current.execute(new DummyProcess(1000,true),pl);
		current.execute(new DummyProcess(1000,true),pl);
		current.execute(new DummyProcess(1000,true),pl);
		current.execute(new DummyProcess(1000,true),pl);
		// Wait til everything should be finished.
		try { Thread.sleep(3000); } catch (InterruptedException ex) { }
		Assert.assertEquals("No tasks queued",0,pl.getQueued());
		Assert.assertEquals("No tasks running",0,pl.getRunning());
		Assert.assertEquals("Max One Thread Running",2,pl.getMaxRunning());
		Assert.assertEquals("All Tasks Finished",4,pl.getFinished());
	}	   
    
	public static void main(String[] args) {
		junit.textui.TestRunner.run(ProcessPoolTest.class);
	}
    

}
