package nl.moj.client.codecompletion;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Code Node : Tree like construct to store types and identifiers with their type to
 *             ease code completion lookup.
 * @author E.Hooijmeijer 
 */

public abstract class CodeNode implements Comparable {

	private static final CodeNode[] NOTHING=new CodeNode[0];

	/** End point. */
	public static class LeafNode extends CodeNode {
		public LeafNode(CodeNode parent,String name,boolean isStatic) {
			super(parent,name,isStatic);
		}
		public CodeNode[] getLeafNodes() {
            return NOTHING;
        }
	}
	
	/** BranchNode : stores package names, types and subtypes. */
	public static class BranchNode extends CodeNode {
		private List mySubNodes=new ArrayList();
		public BranchNode(CodeNode parent,String name,boolean isStatic) {
			super(parent,name,isStatic);			
		}
		public void addNode(CodeNode n) {
			mySubNodes.add(n);
		}
		public CodeNode[] getLeafNodes() {
            return (CodeNode[])mySubNodes.toArray(new CodeNode[mySubNodes.size()]);
        }

	}
	
	/** ReferenceNode : something that has a name and refers to another type. Fields, Methods and such. */ 	
	public static class ReferenceNode extends CodeNode {
		private CodeNode reference;
		public ReferenceNode(CodeNode parent,String name,CodeNode node,boolean isStatic) {
			super(parent,name,isStatic);
			reference=node;
		}
		public CodeNode[] getLeafNodes() {
            return reference.getLeafNodes();
        }
		public void find(String nodeName,boolean partial,boolean isStatic,Set results) {
			//
			if (match(nodeName,partial,isStatic)) {
				results.add(this);
			}
			//
			// Do not traverse subnodes.
			// These are already somewhere else in the list.
			//
		}
		protected void render(StringBuffer sb, int indent) {
			for (int t=0;t<indent;t++) sb.append("| ");
			sb.append("+-");
			sb.append(getName()+" --> "+reference.getFullPath());
			sb.append("\n");
        }
	}


	private String myName;
	private boolean myIsStatic;
	private CodeNode myParent;

	public CodeNode(CodeNode parent,String name,boolean isStatic) {
		myName=name;
		myIsStatic=isStatic;
		myParent=parent;
	}
	
	/** returns the name of this node */
	public String getName() {
		return myName;
	}
	
	/** returns true if this branch has children */
	public boolean isBranch() {
		return (getLeafNodes().length>0);
	}

	/** returns the leafnodes if any */
	public abstract CodeNode[] getLeafNodes();
	
	public CodeNode contains(String name) {
		CodeNode[] cn=getLeafNodes();
		for (int t=0;t<cn.length;t++) {
			if (cn[t].getName().equals(name)) return cn[t];
		} 
		return null;
	}
	
	/** 
	 * searches the tree from this node onwards doing a complete or partial match on the name.
	 * @param nodeName the nodename you're looking for.
	 * @param partial if partial matches should be returned as well.
	 * @param isStatic if true, only static nodes will be returned. 
	 * @param results a list to store the results in.
	 */
	public void find(String nodeName,boolean partial,boolean isStatic,Set results) {
		//
		if (match(nodeName,partial,isStatic)) {
			results.add(this);
		}
		//
		CodeNode[] cn=getLeafNodes();
		for (int t=0;t<cn.length;t++) {
			cn[t].find(nodeName,partial,isStatic,results);
		} 
		//		
	}

	/**
	 * returns true if the specified nodeName matches the current node name.
	 * @param nodeName the nodename you're looking for.
	 * @param partial if true, partial matches return true as well.
	 * @param isStatic if true, only static nodes will be returned.
	 */
	public boolean match(String nodeName,boolean partial,boolean isStatic) {
		//
		// Only match non static nodes if isStatic is false.  
		//
		if (this.myIsStatic) {
			if (!isStatic) return false;
		} 
		//
		// Use the myName property directly, instead of the accessor,
		// which may delegate to somewhere else in the tree.
		//
		if (partial) {
			return myName.startsWith(nodeName);
		} else {
			return myName.equals(nodeName);
		}
	}
	
	public CodeNode getParent() {
		return myParent;
	}
	
	public String toString() {
		return "CodeNode("+myName+")";
	}
	
	public int compareTo(Object o) {		
    	return ((CodeNode)o).getName().compareTo(myName);
	}
	
	
	public String render() {
		StringBuffer sb=new StringBuffer();
		render(sb,0);
		return sb.toString(); 
	}
	
	protected void render(StringBuffer sb,int indent) {
		for (int t=0;t<indent;t++) sb.append("| ");
		sb.append("+-");
		sb.append(myName);
		sb.append("\n");
		CodeNode[] cn=getLeafNodes();
		for (int t=0;t<cn.length;t++) {
			cn[t].render(sb,indent+1);
		}
	}
	
	public String getFullPath() {
		//
		CodeNode p=this.getParent();
		if (p!=null) {
			String fp=p.getFullPath();
			if (fp.length()==0) return getName();
						   else return fp+"."+getName();
		} else { 		
			return getName(); 
		}
	}
 	
	
	

}
